/**************************************************************************//**
 * @file		hdj2534.h
 * @brief		Header file for libhdj2534.so shared library
 * @date		2013-07-26
 * @copyright	Hatteland Display AS
 ******************************************************************************/

#ifndef HDJ2534_H
#define	HDJ2534_H

#ifdef SWDL_BUILD
#define SWDL_DELAY 500
#endif

#ifdef _WIN32_WINNT
#define EXPORTED_OBJ extern "C" __declspec(dllexport)
#else
#define EXPORTED_OBJ extern "C" 
#endif

#define MAX_PASSTHRU_MSG_DATA_SIZE	4128 	/**< Maximal size of Pass-Thru message */
#define INDEX_PASSTHRU_MSG_DATA		4		/**< Where data bytes are stored */
#define J2534_API_VERSION_STRING	"04.04" /**< API version string */
#define J2534_DEFAULT_STRING_LENGTH	80		/**< Default length of API strings */

/** Protocol IDs defines, use in call to PassThruConnect parameter ProtocolID */
namespace J2534
{
	/** @brief Protocol names */
	typedef enum _J2534_PROTOCOL
	{
		J1850VPW		= 0x01,		/**< GM / DaimlerChrysler CLASS2 */
		J1850PWM 		= 0x02,		/**< Ford SCP */
		ISO9141			= 0x03,		/**< ISO9141 and ISO9141-2 */
		ISO14230		= 0x04,		/**< ISO14239-4 (Keyword Protocol 2000) */
		CAN				= 0x05,		/**< Raw CAN (flow control not handled
										by interface) */
		ISO15765		= 0x06,		/**< ISO15765-2 flow control enabled */
		SCI_A_ENGINE	= 0x07,		/**< SAE J2610 (DaimlerChrysler SCI)
										configuration for A engine */
		SCI_A_TRANS		= 0x08,		/**< SAE J2610 (DaimlerChrysler SCI)
										configuration for A transmission */
		SCI_B_ENGINE	= 0x09,		/**< SAE J2610 (DaimlerChrysler SCI)
										configuration for B engine */
		SCI_B_TRANS		= 0x0A,		/**< SAE J2610 (DaimlerChrysler SCI) */
	} J2534_PROTOCOL;

	/** @brief Functions' return codes */
	typedef enum _J2534_ERROR_CODE
	{
		STATUS_NOERROR,				/**< Function called successful */
		ERR_NOT_SUPPORTED,			/**< Function not supported */
		ERR_INVALID_CHANNEL_ID,		/**< Invalid ChannelID value */
		ERR_INVALID_PROTOCOL_ID,	/**< Invalid ProtocolID value */
		ERR_NULL_PARAMETER,			/**< NULL pointer supplied where a
										valid pointer is required */
		ERR_INVALID_IOCTL_VALUE,	/**< Invalid value for Ioctl parameter */
		ERR_INVALID_FLAGS,			/**< Invalid flag values */
		ERR_FAILED,					/**< Undefined error, get description
										with PassThruGetLastError */
		ERR_DEVICE_NOT_CONNECTED,	/**< Device ID invalid */
		ERR_TIMEOUT,				/**< Timeout. No message available to
										read or could not read/write the
										specified number of messages. The
										actual number of messages read is
										placed in NumMsgs */
		ERR_INVALID_MSG,			/**< Invalid message structure pointed
										to by pMsg */
		ERR_INVALID_TIME_INTERVAL,	/**< Invalid TimeInterval value */
		ERR_EXCEEDED_LIMIT,			/**< Exceeded maximum number of message
										IDs or allocated space */
		ERR_INVALID_MSG_ID,			/**< Invalid MsgID value */
		ERR_DEVICE_IN_USE,			/**< Device is currently open */
		ERR_INVALID_IOCTL_ID,		/**< Invalid IoctlID value */
		ERR_BUFFER_EMPTY,			/**< Protocol message buffer empty */
		ERR_BUFFER_FULL,			/**< Protocol message buffer full */
		ERR_BUFFER_OVERFLOW,		/**< Protocol message buffer overflow */
		ERR_PIN_INVALID,			/**< Invalid pin number */
		ERR_CHANNEL_IN_USE,			/**< Channel already in use */
		ERR_MSG_PROTOCOL_ID,		/**< Protocol type does not match with
										Channel ID protocol */
		ERR_INVALID_FILTER_ID,		/**< Invalid FilterID value */
		ERR_NO_FLOW_CONTROL,		/**< No flow control filer set or
										matched (ISO15765 only) */
		ERR_NOT_UNIQUE,				/** A CAN ID in pPatternMsg or
										pFlowControlMsg matches either ID
										in an existing FLOW_CONTROL_FILTER */
		ERR_INVALID_BAUDRATE,		/**< The desired baud rate cannot be
										achieved within the tolerance */
		ERR_INVALID_DEVICE_ID,		/**< Unable to communication with device */
		// User defined faults
		ERR_CANT_REMOVE_DEVICE,		/**< Unable to remove selected device */
		ERR_CANT_REMOVE_CHANNEL,	/**< Error in disconnect of a channel */
	} J2534_ERROR_CODE;

	/** @brief Filter types */
	typedef enum _J2534_FILTER
	{
		PASS_FILTER			= 0x01,	/**< Allow matching messages into the
										receive queue */
		BLOCK_FILTER		= 0x02,	/**< Keep matching messages out of the
										receive queue */
		FLOW_CONTROL_FILTER	= 0x03,	/** Defines a filter and outgoing flow
										control message to support the
										ISO 15765-2 flow control mechanism */
	} J2534_FILTER;

	/** @brief I/O Control IDs */
	typedef enum _J2534_IOCTL
	{
		GET_CONFIG							= 0x01,	/**< Get the vehicle network config */
		SET_CONFIG							= 0x02,	/**< Set the vehicle network config */
		READ_VBATT							= 0x03,	/**< Read the voltage on pin 16 of the
														J1962(carb) connector */
		FIVE_BAUD_INIT						= 0x04,	/**< Initiate a 5 baud initialization
														sequence */
		FAST_INIT							= 0x05,	/**< Initiate a fast initialization
														sequence */
		CLEAR_TX_BUFFER						= 0x07,	/** Clear all messages in the transmit
														queue */
		CLEAR_RX_BUFFER						= 0x08,	/**< Clear all messages in the receive
														queue */
		CLEAR_PERIODIC_MSGS					= 0x09,	/**< Clear all periodic messages
														(stop all periodic msg transm.) */
		CLEAR_MSG_FILTERS					= 0x0A,	/**< Clear all message filters
														(stop all filtering ) */
		CLEAR_FUNCT_MSG_LOOKUP_TABLE		= 0x0B,	/**< Clear Functional Message Look-up
														Table */
		ADD_TO_FUNCT_MSG_LOOKUP_TABLE		= 0x0C,	/**< Add a functional message address
														to the Functional Message Look-up
														Table */
		DELETE_FROM_FUNCT_MSG_LOOKUP_TABLE	= 0x0D, /**< Delete a functional message
														address from the Functional
														Message Look-up Table */
		READ_PROG_VOLTAGE					= 0x0E,	/**< Read the feedback of the
														programmable voltage set by
														PassThruSetProgrammingVoltage */
	} J2534_IOCTL;

	/** @brief I/O Control Configuration Parameters (for IOCTL_GET/SET_CONFIG) */
	typedef enum _J2534_IOCTL_PARAM
	{
		DATA_RATE = 0x01,		/**< Represents the desired baud rate,
									5 - 500000 baud. No default value. */
		LOOPBACK = 0x03,		/**< 0* = Don't echo tx msg in rx queue,
									1  = Echo tx msg in rx queue. */
		NODE_ADDRESS,           /**< Set J1850PWM physical layer node address. */
		NETWORK_LINE,			/**< Set J1850PWM active network lines
									during communication:
									0* = BUS_NORMAL,
									1  = BUS_PLUS,
									2  = BUS_MINUS. */
		P1_MIN,					/**< ISO9141 ECU response min inter-byte
									time in ms. Default 0 ms. */
		P1_MAX,					/**< ISO9141 ECU response max inter-
									byte time in ms. Default 20 ms. */
		P2_MIN,					/**< ISO9141 min time in ms between tester request
									& ECU response. Default 25 ms. */
		P2_MAX,					/**< ISO9141 max time in ms between tester request
									& ECU response. Default 25 ms.*/
		P3_MIN,					/**< ISO9141 min time in ms between end of ECU
									response & new tester response Default 55 ms. */
		P3_MAX,					/**< ISO9141 max time in ms between end of ECU
									response & new tester response. Default 5000 ms. */
		P4_MIN,					/**< ISO9141 tester request min inter-byte time
									in ms.Default 5 ms. */
		P4_MAX,					/**< ISO9141 tester request max inter-byte time
									in ms. Default 20 ms. */
		W1,						/**< ISO9141 max time in ms from end of the
									address byte to start of sync pattern.
									Default 300 ms. */
		W2,						/**< ISO9141 max time in ms from end of sync
									pattern to start of key byte 1. Default 20 ms. */
		W3,						/**< ISO9141 max time in ms between key byte 1
									and key byte 2. Default 20 ms. */
		W4,						/**< ISO9141 max time in ms between key byte 2 and
									inversion from tester. Default 50 ms. */
		W5,						/**< ISO9141 max time in ms before the tester
									start to transmit address byte. Default 300 ms. */
		TIDLE,					/**< ISO9141 amount of bus idle time ms needed
									before fast initialization. Default is value of W5. */
		TINIL,					/**< ISO9141 duration in ms for low pulse in
									fast initialization. Default 25 ms. */
		TWUP,					/**< ISO9141 duration in ms of wake-up pulse
									in fast initialization. Default 50 ms.*/
		PARITY,					/**< ISO9141 parity type for detecting bit errors:
									0* = NO_PARITY,
									1  = ODD_PARITY,
									2  = EVEN_PARITY. */
		BIT_SAMPLE_POINT,		/**< Set CAN bit sample point 0 - 100%. Default 80. */
		SYNC_JUMP_WIDTH,		/**< Set CAN sync jump width as percent of the bit
									time 0-100%. Default 15. */
		W0,						/**< ISO9141 min bus idle time in ms before the tester
									starts to transmit the address byte. Default 300 ms. */
		T1_MAX,					/**< SCI_A_ENGINE, SCI_B_ENGINE, SCI_A_TRANS or SCI_B_TRANS max
									inter-frame response delay in ms. Default 20 ms. */
		T2_MAX,					/**< SCI_A_ENGINE, SCI_B_ENGINE, SCI_A_TRANS or SCI_B_TRANS max
									inter-frame request delay in ms. Default 100 ms. */
		T4_MAX,					/**< SCI_A_ENGINE, SCI_B_ENGINE, SCI_A_TRANS or SCI_B_TRANS max
									inter-message response delay in ms. Default 20 ms. */
		T5_MAX,					/**< SCI_A_ENGINE, SCI_B_ENGINE, SCI_A_TRANS or SCI_B_TRANS max
									inter-message request delay in ms. Default 100 ms. */
		ISO15765_BS,			/**< ISO15765 block size for segmented transfer. Default 0
									(may be overridden by firmware). */
		ISO15765_STMIN,			/**< ISO15765 separation time for segmented transfer. Default 0
									(may be overridden by firmware). */
		DATA_BITS,				/**< ISO9141, ISO14230:
									0* = 8 data bits,
									1  = 7 data bits. */
		FIVE_BAUD_MOD,			/**< ISO9141, ISO14230:
									0* = Init as defined in ISO9141-2 and ISO14230-4,
									1  = ISO9141 init followed by interface sending inverted Key Byte 2,
									2  = ISO9141 init followed by ECU sending inverted address,
									3  = Init as defined in ISO9141. */
		BS_TX,					/**< ISO15765, block size to be used to transmit segmented messages */
		STMIN_TX,				/**< ISO15765, separation time to be used to transmit
									segmented messages */
		T3_MAX,					/**< SCI_A_ENGINE, SCI_B_ENGINE, SCI_A_TRANS or SCI_B_TRANS max
									response delay from the ECU in ms. Default 100 ms. */
		ISO15765_WFT_MAX		/**< ISO15765, Number of WAIT flow control frames allowed during a
									multi segment transfer */

	} J2534_IOCTL_PARAM;

	/** @brief SOCONFIG structure definition */
	typedef struct
	{
		unsigned long Parameter; /**< name of parameter */
		unsigned long Value; /**< Value of parameter */
	} SCONFIG;

	/** @brief SCONFIG_LIST structure definition */
	typedef struct
	{
		unsigned long NumOfParams; /**< number of SCONFIG elements */
		SCONFIG *ConfigPtr; /**< array of SCONFIG */
	} SCONFIG_LIST;

	#pragma pack(1)
	/** @brief Structure which can be used by the 'Flags' field in PassThruConnect() */
	typedef struct _J2534_ConnectFlags
	{
		union {
			struct {
				unsigned char : 8; /**< Reserved for SAE */
				unsigned char Can29BitId : 1;
				unsigned char ISO9141_NoChecksum : 1;
				unsigned char : 1; /**< Reserved for SAE */
				unsigned char CanIdBoth : 1;
				unsigned char ISO9141_KlineOnly : 1;
				unsigned char : 3; /**< Reserved for SAE */
				unsigned char : 8; /**< Reserved for SAE J2534-2 */
				unsigned char : 4; /**< Reserved for tool manufacturer */
				unsigned char FuncAddr : 1; /**< Vendor specific, defined in 109-1152 */
				unsigned char PhysicalChannel : 3; /**< -..- */
			} bits;
			unsigned long value;
		};

	} J2534_ConnectFlags;

	#define PHYSICAL_CHANNEL        (1<<29)
	#define FUNC_ADDR               (1<<28)     // Vendor specific, defined in 109-1152
	#define ISO9141_K_LINE_ONLY     (1<<12)
	#define CAN_ID_BOTH             (1<<11)
	#define ISO9141_NO_CHECKSUM     (1<<9)
	#define CAN_29BIT_ID            (1<<8)

	/** @brief Structure which can be used by the 'RxStatus' field in PASSTHRU_MSG */
	typedef struct _J2534_RxStatus
	{
		union {
			struct {
				unsigned char TxMsgType : 1;
				unsigned char StartOfMessage :1;
				unsigned char RxBreak : 1;
				unsigned char TxIndication : 1;
				unsigned char ISO15765_PaddingError : 1;
				unsigned char : 2; /**< Reserved for SAE */
				unsigned char ISO15765_AddrType : 1;
				unsigned char Can29BitId : 1;
				unsigned char : 1; /**<Reserved for SAE */
				unsigned char CanFrameRaw : 1; /**< Vendor specific, defined in 109-1152 */
				unsigned char : 5; /**< Reserved for SAE */
				unsigned char : 8; /**< Reserved for SAE J2534-2 */
				unsigned char : 8; /**< Reserved for tool manufacturer */
			} bits;
			unsigned long value;
		};
	} J2534_RxStatus;

	#define CAN_FRAME_RAW           (1<<10)     // Vendor specific, defined in 109-1152
	#define ISO15765_ADDR_TYPE      (1<<7)
	#define ISO15765_PADDING_ERROR  (1<<4)
	#define TX_INDICATION           (1<<3)
	#define RX_BREAK                (1<<2)
	#define START_OF_MESSAGE        (1<<1)
	#define TX_MSG_TYPE             (1<<0)

	/** @brief Structure which can be used by the 'TxFlags' field in PASSTHRU_MSG */
	typedef struct _J2534_TxFlags
	{
		union {
			struct {
				unsigned char : 6;	/**< Reserved for SAE */
				unsigned char ISO15765_FramePad : 1;
				unsigned char ISO15765_AddrType : 1;
				unsigned char Can29BitId : 1;
				unsigned char WaitP3MinOnly : 1;
				unsigned char CanFrameRaw : 1; /**< Vendor specific, defined in 109-1152 */
				unsigned char : 5; /**< Reserved for SAE */
				unsigned char : 6; /**< Reserved for SAE J2534-2 */
				unsigned char SciMode : 1;
				unsigned char SciTxVoltage : 1;
				unsigned char : 8; /**< Reserved for tool manufacturer */
			} bits;
			unsigned long value;
		};
	} J2534_TxFlags;

	#define SCI_TX_VOLTAGE          (1<<23)
	#define SCI_MODE                (1<<22)
	#define WAIT_P3_MIN_ONLY        (1<<9)
	#define ISO15765_FRAME_PAD      (1<<6)

	#pragma pack()
	/** @brief Passthru message structure */
	struct PASSTHRU_MSG
	{
		unsigned long ProtocolID; /**< Vehicle network protocol */
		unsigned long RxStatus; /**< Receive message status (using J2534_RxStatus structure) */
		unsigned long TxFlags; /**< Transmit message flags (using J2534_TxFlags structure) */
		unsigned long Timestamp; /**< Receive message timestamp (in microseconds) */
		unsigned long DataSize; /**< Byte size of message payload in the data array */
		unsigned long ExtraDataIndex; /**< Start of extra data */
		unsigned char Data[MAX_PASSTHRU_MSG_DATA_SIZE]; /**< Message payload or data */
	};
#ifndef APP_BUILD
	EXPORTED_OBJ J2534_ERROR_CODE PassThruOpen(
	  void            *pName,
	  unsigned long   *pDeviceID);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruClose(
	  unsigned long   DeviceID);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruConnect(
	  unsigned long   DeviceID,
	  J2534_PROTOCOL  ProtocolID,
	  unsigned long   Flags,              // Using J2534_ConnectFlags structure
	  unsigned long   BaudRate,
	  unsigned long   *pChannelID);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruDisconnect(
	  unsigned long   ChannelID);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruReadMsgs(
	  unsigned long   ChannelID,
	  PASSTHRU_MSG    *pMsg,
	  unsigned long   *pNumMsgs,
	  unsigned long   Timeout);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruWriteMsgs(
	  unsigned long   ChannelID,
	  PASSTHRU_MSG    *pMsg,
	  unsigned long   *pNumMsgs,
	  unsigned long   Timeout);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruStartPeriodicMsg(
	  unsigned long   ChannelID,
	  PASSTHRU_MSG    *pMsg,
	  unsigned long   *pMsgID,
	  unsigned long   TimeInterval);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruStopPeriodicMsg(
	  unsigned long   ChannelID,
	  unsigned long   MsgID);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruStartMsgFilter(
	  unsigned long   ChannelID,
	  J2534_FILTER    FilterType,
	  PASSTHRU_MSG    *pMaskMsg,
	  PASSTHRU_MSG    *pPatternMsg,
	  PASSTHRU_MSG    *pFlowControlMsg,
	  unsigned long   *pFilterID);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruStopMsgFilter(
	  unsigned long   ChannelID,
	  unsigned long   FilterID);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruSetProgrammingVoltage(
	  unsigned long   DeviceID,
	  unsigned long   PinNumber,
	  unsigned long   Voltage);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruReadVersion(
	  unsigned long   DeviceID,
	  char            *pFirmwareVersion,
	  char            *pLibVersion,
	  char            *pApiVersion);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruGetLastError(
	  char            *pErrorDescription);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruIoctl(
	  unsigned long   ChannelID,
	  J2534_IOCTL     IoctlID,
	  void            *pInput,
	  void            *pOutput);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruGetStatus(
	  unsigned long   ChannelID,
	  unsigned long   *status);

	EXPORTED_OBJ J2534_ERROR_CODE PassThruReset(void);
#endif
}

#ifdef SWDL_BUILD
namespace SWDL
{
	EXPORTED_OBJ bool SWDLReadMem(
	  void 			*data,
	  unsigned int 	len,
	  unsigned int 	addr);

	EXPORTED_OBJ bool SWDLWriteMem(
	  void 			*data,
	  unsigned int 	len,
	  unsigned int 	addr);

	EXPORTED_OBJ bool SWDLEraseFlash(
	  unsigned int 	startSector,
	  unsigned int 	endSector);

	EXPORTED_OBJ bool SWDLCall(
	  unsigned int 	addr);

	EXPORTED_OBJ bool SWDLStop();
}
#endif

#endif	/* HDJ2534 */
